const nodemailer = require('nodemailer');
const fs = require('fs');
const path = require('path');

class EmailService {
  constructor() {
    console.log('[EMAIL] ===== INITIALIZING EMAIL SERVICE =====');
    
    try {
      const smtpHost = process.env.EMAIL_SMTP_HOST;
      const smtpUser = process.env.EMAIL_SMTP_USER;
      const smtpPassword = process.env.EMAIL_SMTP_PASSWORD;
      const smtpPortValue = process.env.EMAIL_SMTP_PORT;
      const smtpPort = smtpPortValue ? Number(smtpPortValue) : 587;
      const smtpSecure = String(process.env.EMAIL_SMTP_SECURE || '').toLowerCase() === 'true';
      
      if (!smtpHost || !smtpUser || !smtpPassword) {
        console.warn('[EMAIL] WARNING: SMTP settings missing, email service will be disabled');
        this.transporter = null;
        return;
      }
      
      if (Number.isNaN(smtpPort)) {
        console.warn('[EMAIL] WARNING: Invalid EMAIL_SMTP_PORT, using default 587');
      }
      
      // Initialize nodemailer transporter
      try {
        this.smtpConfig = {
          host: smtpHost,
          port: Number.isNaN(smtpPort) ? 587 : smtpPort,
          secure: smtpSecure, // true for 465, false for other ports
          auth: {
            user: smtpUser,
            pass: smtpPassword
          },
          // Enable debug logging
          debug: true,
          logger: true
        };
        
        this.transporter = nodemailer.createTransport(this.smtpConfig);
        
        // Add event listeners for SMTP connection logging
        this.transporter.on('token', (token) => {
          console.log('[EMAIL] [SMTP] OAuth2 token received:', token);
        });
        
        // Log SMTP connection events
        this.transporter.on('idle', () => {
          console.log('[EMAIL] [SMTP] Connection is idle');
        });
        
        this.transporter.on('error', (error) => {
          console.error('[EMAIL] [SMTP] Connection error:', error.message);
          console.error('[EMAIL] [SMTP] Error code:', error.code || 'N/A');
          console.error('[EMAIL] [SMTP] Error stack:', error.stack || 'N/A');
        });
        
        console.log('[EMAIL] Nodemailer transporter initialized successfully');
        console.log('[EMAIL] SMTP Host:', this.smtpConfig.host);
        console.log('[EMAIL] SMTP Port:', this.smtpConfig.port);
        console.log('[EMAIL] SMTP Secure (SSL):', this.smtpConfig.secure);
        console.log('[EMAIL] SMTP User:', this.smtpConfig.auth.user);
        
        // Verify SMTP connection
        console.log('[EMAIL] [SMTP] Verifying SMTP connection...');
        this.transporter.verify((error, success) => {
          if (error) {
            console.error('[EMAIL] [SMTP] ===== SMTP CONNECTION VERIFICATION FAILED =====');
            console.error('[EMAIL] [SMTP] Error message:', error.message);
            console.error('[EMAIL] [SMTP] Error code:', error.code || 'N/A');
            console.error('[EMAIL] [SMTP] Error command:', error.command || 'N/A');
            console.error('[EMAIL] [SMTP] Error response:', error.response || 'N/A');
            console.error('[EMAIL] [SMTP] Error responseCode:', error.responseCode || 'N/A');
            if (error.responseCode === 535) {
              console.error('[EMAIL] [SMTP] Authentication failed - check username and password');
            }
            console.error('[EMAIL] [SMTP] ===============================================');
          } else {
            console.log('[EMAIL] [SMTP] ===== SMTP CONNECTION VERIFIED SUCCESSFULLY =====');
            console.log('[EMAIL] [SMTP] Server is ready to accept messages');
            console.log('[EMAIL] [SMTP] ===============================================');
          }
        });
      } catch (transporterError) {
        console.error('[EMAIL] ERROR: Failed to initialize transporter:', transporterError.message);
        console.error('[EMAIL] ERROR stack:', transporterError.stack || 'N/A');
        this.transporter = null;
        return;
      }
      
      this.from = process.env.EMAIL_FROM || 'noreply@example.com';
      this.subject = process.env.EMAIL_SUBJECT || 'Booking Confirmation';
      
      // Load email template
      this.templatePath = process.env.EMAIL_TEMPLATE_PATH || './letter.html';
      if (fs.existsSync(this.templatePath)) {
        this.emailTemplate = fs.readFileSync(this.templatePath, 'utf8');
        console.log('[EMAIL] Email template loaded from:', this.templatePath);
      } else {
        console.warn('[EMAIL] WARNING: Email template not found at:', this.templatePath);
        this.emailTemplate = null;
      }
      
      console.log('[EMAIL] ===== EMAIL SERVICE INITIALIZED SUCCESSFULLY =====');
    } catch (error) {
      console.error('[EMAIL] ===== EMAIL SERVICE INITIALIZATION FAILED =====');
      console.error('[EMAIL] Error type:', error.constructor.name);
      console.error('[EMAIL] Error message:', error.message);
      if (error.stack) {
        console.error('[EMAIL] Error stack:', error.stack);
      }
      console.error('[EMAIL] =================================================');
      this.transporter = null;
      this.emailTemplate = null;
    }
  }

  /**
   * Send confirmation email to customer
   * @param {string} to - Recipient email address
   * @param {Object} sessionData - Stripe session data
   * @returns {Promise<Object>} Email send result
   */
  async sendConfirmationEmail(to, sessionData) {
    console.log('[EMAIL] ===== ATTEMPTING TO SEND EMAIL =====');
    console.log('[EMAIL] Timestamp:', new Date().toISOString());
    
    if (!this.transporter) {
      const error = new Error('Email service not initialized - check SMTP configuration');
      console.error('[EMAIL] ERROR: Transporter is not initialized');
      throw error;
    }

    if (!to || !to.includes('@')) {
      const error = new Error('Invalid recipient email address');
      console.error('[EMAIL] ERROR: Invalid recipient email:', to);
      throw error;
    }

    try {
      // Build email content
      const emailContent = this.buildEmailContent(sessionData);
      
      const mailOptions = {
        from: this.from || 'noreply@example.com',
        to: to,
        subject: this.subject || 'Booking Confirmation',
        html: emailContent
      };

      console.log('[EMAIL] Email details:');
      console.log('[EMAIL]   From:', mailOptions.from);
      console.log('[EMAIL]   To:', to);
      console.log('[EMAIL]   Subject:', mailOptions.subject);
      console.log('[EMAIL]   Session ID:', sessionData?.id || 'N/A');
      console.log('[EMAIL] [SMTP] Attempting to connect to SMTP server...');
      console.log('[EMAIL] [SMTP] Host:', this.smtpConfig?.host);
      console.log('[EMAIL] [SMTP] Port:', this.smtpConfig?.port);
      
      const sendStartTime = Date.now();
      const info = await this.transporter.sendMail(mailOptions);
      const sendDuration = Date.now() - sendStartTime;
      
      console.log('[EMAIL] ===== EMAIL SENT SUCCESSFULLY =====');
      console.log('[EMAIL] Send duration:', sendDuration, 'ms');
      console.log('[EMAIL] Message ID:', info.messageId);
      console.log('[EMAIL] [SMTP] Server response:', info.response || 'N/A');
      console.log('[EMAIL] [SMTP] Accepted recipients:', info.accepted || []);
      console.log('[EMAIL] [SMTP] Rejected recipients:', info.rejected || []);
      console.log('[EMAIL] [SMTP] Pending recipients:', info.pending || []);
      if (info.responseCode) {
        console.log('[EMAIL] [SMTP] Response code:', info.responseCode);
      }
      if (info.envelope) {
        console.log('[EMAIL] [SMTP] Envelope from:', info.envelope.from);
        console.log('[EMAIL] [SMTP] Envelope to:', info.envelope.to);
      }
      console.log('[EMAIL] ====================================');
      
      return info;
    } catch (error) {
      console.error('[EMAIL] ===== EMAIL SEND FAILED =====');
      console.error('[EMAIL] Error type:', error.constructor.name);
      console.error('[EMAIL] Error message:', error.message);
      console.error('[EMAIL] Error code:', error.code || 'N/A');
      console.error('[EMAIL] Error command:', error.command || 'N/A');
      console.error('[EMAIL] [SMTP] Server response:', error.response || 'N/A');
      console.error('[EMAIL] [SMTP] Response code:', error.responseCode || 'N/A');
      if (error.responseCode === 535) {
        console.error('[EMAIL] [SMTP] Authentication failed - check username and password');
      } else if (error.responseCode === 550) {
        console.error('[EMAIL] [SMTP] Mailbox unavailable - check recipient email address');
      } else if (error.code === 'ECONNREFUSED') {
        console.error('[EMAIL] [SMTP] Connection refused - check SMTP host and port');
      } else if (error.code === 'ETIMEDOUT') {
        console.error('[EMAIL] [SMTP] Connection timeout - check network connectivity');
      } else if (error.code === 'EAUTH') {
        console.error('[EMAIL] [SMTP] Authentication error - check username and password');
      }
      if (error.stack) {
        console.error('[EMAIL] Error stack:', error.stack);
      }
      console.error('[EMAIL] =============================');
      throw error;
    }
  }

  /**
   * Build email content from template and session data
   * @param {Object} sessionData - Stripe session data
   * @returns {string} HTML email content
   */
  buildEmailContent(sessionData) {
    // If template exists, use it and replace placeholders
    if (this.emailTemplate) {
      let content = this.emailTemplate;
      
      // Replace common placeholders with session data
      const replacements = {
        '{{SESSION_ID}}': sessionData?.id || 'N/A',
        '{{AMOUNT}}': sessionData?.amount_total ? `$${(sessionData.amount_total / 100).toFixed(2)}` : 'N/A',
        '{{CURRENCY}}': sessionData?.currency?.toUpperCase() || 'USD',
        '{{CUSTOMER_EMAIL}}': sessionData?.customer_email || 'N/A',
        '{{PAYMENT_STATUS}}': sessionData?.payment_status || 'N/A',
        '{{CRUISE_TITLE}}': sessionData?.metadata?.cruise_title || 'Cruise Booking',
        '{{SHIP_NAME}}': sessionData?.metadata?.ship_name || 'N/A',
        '{{SAIL_DATE}}': sessionData?.metadata?.sail_date || 'N/A',
        '{{GUEST_NAME}}': sessionData?.metadata?.guest_name || 'Guest',
        '{{PACKAGE_ID}}': sessionData?.metadata?.package_id || 'N/A',
        '{{DATE}}': new Date().toLocaleDateString('en-US', { 
          year: 'numeric', 
          month: 'long', 
          day: 'numeric' 
        })
      };
      
      // Replace all placeholders
      Object.keys(replacements).forEach(placeholder => {
        content = content.replace(new RegExp(placeholder, 'g'), replacements[placeholder]);
      });
      
      return content;
    }
    
    // Fallback: simple HTML template
    return `
      <!DOCTYPE html>
      <html>
      <head>
        <meta charset="utf-8">
        <style>
          body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
          .container { max-width: 600px; margin: 0 auto; padding: 20px; }
          .header { background-color: #f4f4f4; padding: 20px; text-align: center; }
          .content { padding: 20px; }
          .footer { background-color: #f4f4f4; padding: 20px; text-align: center; font-size: 12px; color: #666; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="header">
            <h1>Thank You!</h1>
          </div>
          <div class="content">
            <p>Your booking has been confirmed.</p>
            <p><strong>Booking Details:</strong></p>
            <ul>
              <li>Session ID: ${sessionData?.id || 'N/A'}</li>
              <li>Amount: ${sessionData?.amount_total ? `$${(sessionData.amount_total / 100).toFixed(2)}` : 'N/A'} ${sessionData?.currency?.toUpperCase() || 'USD'}</li>
              <li>Payment Status: ${sessionData?.payment_status || 'N/A'}</li>
            </ul>
            ${sessionData?.metadata?.cruise_title ? `<p><strong>Cruise:</strong> ${sessionData.metadata.cruise_title}</p>` : ''}
            ${sessionData?.metadata?.ship_name ? `<p><strong>Ship:</strong> ${sessionData.metadata.ship_name}</p>` : ''}
            ${sessionData?.metadata?.sail_date ? `<p><strong>Sail Date:</strong> ${sessionData.metadata.sail_date}</p>` : ''}
            <p>You will receive your electronic tickets via email 14 days before the start of the cruise.</p>
          </div>
          <div class="footer">
            <p>Thank you for choosing us!</p>
          </div>
        </div>
      </body>
      </html>
    `;
  }

  /**
   * Test email service
   * @param {string} testEmail - Email address to send test to
   * @returns {Promise<Object>} Test result
   */
  async sendTestEmail(testEmail) {
    if (!this.transporter) {
      throw new Error('Email service not initialized');
    }

    const testSessionData = {
      id: 'test_session_123',
      amount_total: 10000, // $100.00
      currency: 'usd',
      payment_status: 'paid',
      customer_email: testEmail,
      metadata: {
        cruise_title: 'Test Cruise',
        ship_name: 'Test Ship',
        sail_date: '2025-12-31',
        guest_name: 'Test Guest',
        package_id: 'TEST123'
      }
    };

    return await this.sendConfirmationEmail(testEmail, testSessionData);
  }
}

module.exports = EmailService;

